# Created by David Bickel on 17 August 2007.
# setMethod("[", signature(x = "Matrix"... changed on 1 Feb. 2008.
# setReplaceMethod("[", "Matrix"... added on 3 April 2008, after Zahra Montazeri's logb changes.
# setReplaceMethod("[", "Numeric"... added on 3 April 2008, after Zahra Montazeri's logb changes.
# "plot", signature(x = "numeric", y = "biasEstimate") changed 9 April 2008.
#  More changes 16 April 2008.

library(Biobase)

removeMethods("plot") # added 2 July 2008
removeMethods("stripchart") # added 20 Aug. 2008
removeMethods("summary") # added 8 January 2009
removeMethods("print")
removeMethods("mean")

setMethod("plot", signature(x = "numeric", y = "list"), function(x, y, ...)
{
	plot(x = x, y = data.frame(lapply(y, as, Class = "numeric")), ...)
})
setMethod("plot", signature(x = "numeric", y = "data.frame"), function(x, y, col, pch, type, legend.legend, legend.x, xlim, ylim, normalizeFUN, MARGIN, ...)
{
	stopifnot(length(x) == nrow(y))
	if(missing(MARGIN))
		MARGIN <- 1
	if(!missing(normalizeFUN) && is.function(normalizeFUN))
	{
		if(MARGIN == 1)
		{
			for(i in 1:nrow(y))
				y[i, ] <- y[i, ] / normalizeFUN(y[i, ])
		}
		else if(MARGIN == 2)
		{
			for(j in 1:ncol(y))
				y[, j] <- y[, j] / normalizeFUN(y[, j])
		}
		else
			stop("bad MARGIN")
	}
	if(missing(col))
		col <- 1 # :length(x)
	if(missing(pch))
		pch <- 1:length(x)
	if(missing(type))
		type <- default("p", "type")
	if(missing(legend.x))
		legend.x <- default("center", "legend.x")
	if(length(col) == 1)
		col <- rep(col, length(x))
	if(length(pch) == 1)
		pch <- rep(pch, length(x))
	nam <- colnames(y)
	if(missing(legend.legend))
		legend.legend <- default(if(is.null(nam)) col else nam, "legend.legend")
	if(missing(xlim))
		xlim <- range(x, na.rm = TRUE)
	if(missing(ylim))
		ylim <- range(as.numeric(as.matrix(y[x >= xlim[1] & x <= xlim[2], ])), na.rm = TRUE)
	plot(x = x, y = y[[1]], col = col[1], pch = pch[1], type = type, xlim = xlim, ylim = ylim, ...)
	fun <- if(type == "p")
		points
	else if(type == "l")
		lines
	else
		stop("bad type")
	for(j in 2:ncol(y))
		fun(x = x, y = y[, j], col = col[j], pch = pch[j], ...)
	legend(x = legend.x, legend = legend.legend, col = col, pch = pch)
})

setClass("pNumeric", representation("numeric"))
setValidity("pNumeric", function(object)
{
	num.ok <- all(is.na(object)) || all(object >= 0 & object <= 1, na.rm = TRUE)
	if(!num.ok)
	{ printInvalid(object); browser()}
	num.ok
})
setAs(from = "pNumeric", to = "numeric", function(from)
{
  x <- from@.Data
  names(x) <- names(from)
  x
})
setMethod("[", signature(x = "pNumeric", i = "ANY", j = "missing"), function(x, i, j, drop)
{
  pNumeric(as(x, "numeric")[i])
})
setReplaceMethod("[", signature(x = "pNumeric", i = "ANY", j = "missing"), function(x, i, j, value) # showMethods("[<-")
{
	vec <- as(x, "numeric")
	tr <- try(vec[i] <- value)
	if(is(tr, "try-error") || length(vec) != length(x))
	{ message("bad replacement of ", class(x)); browser()}
	pNumeric(vec)
})

setClass("Numeric", representation("numeric"))
setValidity("Numeric", function(object)
{
	num.ok <- all(is.na(object)) || all(object >= 0, na.rm = TRUE)
	if(!num.ok)
	{ printInvalid(object); browser()}
	num.ok
})
setAs(from = "Numeric", to = "numeric", function(from)
{
  x <- from@.Data
  names(x) <- names(from)
  x
})
setMethod("[", signature(x = "Numeric", i = "ANY", j = "missing"), function(x, i, j, drop)
{
  Numeric(as(x, "numeric")[i])
})
setReplaceMethod("[", signature(x = "Numeric", i = "ANY", j = "missing"), function(x, i, j, value) # showMethods("[<-")
{
	vec <- as(x, "numeric")
	tr <- try(vec[i] <- value)
	if(is(tr, "try-error") || length(vec) != length(x))
	{ message("bad replacement of ", class(x)); browser()}
	Numeric(vec)
})

setClass("scalar", representation("numeric"))
setValidity("scalar", function(object)
{
	length(object) %in% c(0, 1) # changed 16 April 2008
})
setClass("Scalar", representation("scalar"))
setValidity("Scalar", function(object)
{
	length(object) == 0 || is.na(object) || object >= 0 # changed 16 April 2008
})
setIs("Scalar", "Numeric")

setClass("Matrix", representation("matrix"))
setValidity("Matrix", function(object)
{
	all(as.numeric(object) >= 0, na.rm = TRUE)
})
setMethod("[", signature(x = "Matrix", i = "ANY", j = "missing"), function(x, i, j, drop)
{
  mat <- as(x, "matrix")
  mat <- mat[i, , drop = FALSE]
  Matrix(mat)
})
setMethod("[", signature(x = "Matrix", i = "missing", j = "ANY"), function(x, i, j, drop)
{
  mat <- as(x, "matrix")
  mat <- mat[, j, drop = FALSE]
  Matrix(mat)
})
setMethod("[", signature(x = "Matrix", i = "ANY", j = "ANY"), function(x, i, j, drop)
{
  mat <- as(x, "matrix")
  mat <- mat[i, j, drop = FALSE]
  Matrix(mat)
})
setReplaceMethod("[", signature(x = "Matrix", i = "ANY", j = "missing"), function(x, i, j, value) # showMethods("[<-")
{
#	cat("replacing with "); print(value)
	mat <- as(x, "matrix")
	tr <- try(mat[i, ] <- value)
	if(is(tr, "try-error"))
	{ message("bad replacement"); browser()}
	Matrix(mat)
})

setClass("xprnSet", representation(es = "ExpressionSet"))
#setIs("xprnSet", "ExpressionSet")
setAs(from = "xprnSet", to = "ExpressionSet", function(from)
{
  from@es
})
setValidity("xprnSet", function(object)
{
	all(colnames(exprs(object)) == rownames(pData(object)))
})
setMethod("names", signature(x = "xprnSet"), function(x)
{
	featureNames(x)
})
setMethod("print", signature(x = "xprnSet"), function(x)
{
  message(class(x), "; high-throughput data:\n")
  print(as(x, "ExpressionSet"))
})
setMethod("exprs", signature(object = "xprnSet"), function(object)
{
  exprs(as(object, "ExpressionSet"))
})
setMethod("featureNames", signature(object = "xprnSet"), function(object)
{
  featureNames(as(object, "ExpressionSet"))
})
setMethod("pData", signature(object = "xprnSet"), function(object)
{
  pData(as(object, "ExpressionSet"))
})
setMethod("[", signature(x = "xprnSet", i = "ANY", j = "missing"), function(x, i, j, drop)
{
  es <- as(x, "ExpressionSet")
  x@es <- es[i, ]
  x
})
setMethod("[", signature(x = "xprnSet", i = "missing", j = "ANY"), function(x, i, j, drop)
{
  es <- as(x, "ExpressionSet")
  x@es <- es[, j]
  x
})
setMethod("[", signature(x = "xprnSet", i = "ANY", j = "ANY"), function(x, i, j, drop)
{
  es <- as(x, "ExpressionSet")
  x@es <- es[i, j]
  x
})
setMethod("logb", signature(x = "ExpressionSet", base = "missing"), function(x, base)
{
  exprs(x) <- logb(exprs(x))
  x
})
setMethod("logb", signature(x = "xprnSet", base = "missing"), function(x, base)
{
  x@es <- logb(as(x, "ExpressionSet"))
  x
})
# setMethod("plot", signature(x = "xprnSet", y = "missing"), function(x, ...) # see estimated.r
setMethod("stripchart", signature(x = "xprnSet"), function(x, file, factor.name, ...)
{
	if(missing(file))
	{
		file <- paste(annotation(x), factor.name, "pdf", sep = ".")
		message("file = ", file)
	}
	if(missing(factor.name))
	{
		stop("not yet implemented for missing factor.name")
		fac <- factor()
		levs <- "level"
		message("factor.name missing")
	}
	else
	{
		fac <- pData(x)[, factor.name]
		stopifnot(is.factor(fac))
		levs <- levels(fac)
	}
	call.pdf <- is.character(file) && length(file) == 1
	if(call.pdf) pdf(file = file)
	par(mfrow = c(2, 2))
	for(feature in featureNames(x))
	{
		y <- exprs(x[feature, ])
		boo <- is.finite(y)
		if(any(boo))
		{
			y <- y[boo]
			fac <- fac[boo]
			if(length(y) != length(fac))
			{ message("bad length of y or fac"); browser()}
			stripchart(y ~ fac, main = feature, xlab = feature, ...)
		}
		else
			message("data not available for ", feature)
	}
	if(call.pdf) dev.off()
})

setClass("XprnSet", representation("xprnSet"))
setValidity("XprnSet", function(object)
{
	all(as.numeric(exprs(object)) >= 0, na.rm = TRUE)
})
setMethod("print", signature(x = "XprnSet"), function(x)
{
  message(class(x), "; ratio or nonnegative intensity data:\n")
  print(as(x, "ExpressionSet"))
})
setMethod("exprs", signature(object = "XprnSet"), function(object)
{
  Matrix(exprs(as(object, "ExpressionSet")))
})
setMethod("logb", signature(x = "XprnSet", base = "missing"), function(x, base)
{
  logb(as(x, "xprnSet"))
})

setClass("xprnSetPair", representation(x = "xprnSet", y = "xprnSet"))
setValidity("xprnSetPair", function(object)
{
	cla.ok <- !is(object@x, "XprnSet") && !is(object@y, "XprnSet")
	fea.x <- featureNames(object@x)
	fea.y <- featureNames(object@y)
	fea.ok <- length(fea.x) == length(fea.x) && is.character(fea.x) && all(fea.x == fea.y)
	ok <- cla.ok && fea.ok
	if(!ok){printInvalid(object); browser()}
	ok
})
setAs(from = "xprnSetPair", to = "xprnSet", function(from)
{
	phenoData <- rbind(pData(from@x), pData(from@y))
	exprs <- cbind(exprs(from@x), exprs(from@y))
	xprnSet(phenoData = phenoData, exprs = exprs)
})
setMethod("annotation", signature(object = "xprnSetPair"), function(object)
{
	paste(annotation(object@x), annotation(object@y), sep = " vs. ")
})
setMethod("featureNames", signature(object = "xprnSetPair"), function(object) # new 24 April 2008; for bias.r
{
  featureNames(object@x)
})
setMethod("logb", signature(x = "xprnSetPair", base = "missing"), function(x, base)
{
	if(is(x@x, "XprnSet"))
	{
		x@x <- logb(x@x)
		x@y <- logb(x@y)
	}
	else
		warning("no log taken")
  x
})

setClassUnion(name = "xprnSetObject", members = c("xprnSet", "xprnSetPair")) # used in estimate.s, biasEstimate

setClass("xprnSetObjects", representation("list")) # new 28 April 2008
setValidity("xprnSetObjects", function(object)
{
	all(sapply(object, is, class2 = "xprnSetObject"))
})

setClass("xprnSetObjectPair", representation(training = "xprnSetObject", test = "xprnSetObject")) # new 24 April 2008; for bias.r
setValidity("xprnSetObjectPair", function(object)
{
	cla.ok <- TRUE
	fea.training <- featureNames(object@training)
	fea.test <- featureNames(object@test)
	fea.ok <- length(fea.training) == length(fea.training) && is.character(fea.training) && all(fea.training == fea.test)
	ok <- cla.ok && fea.ok
	if(!ok){printInvalid(object); browser()}
	ok
})
setMethod("featureNames", signature(object = "xprnSetObjectPair"), function(object) # new 24 April 2008; for bias.r
{
  featureNames(object@test)
})

setClass("smoothSpline", "list")
setValidity("smoothSpline", function(object)
{
	le.ok <- length(object) == 1
	if(!le.ok)
	{ printInvalid(object); browser()}
	ok <- le.ok && names(object) == "s3"
	if(is.na(ok) || !ok) warning("names lost")
	le.ok
})
setMethod("lines", signature(x = "smoothSpline"), function(x, ...)
{
  Lines(x = x, ...)
})
setMethod("plot", signature(x = "smoothSpline", y = "missing"), function(x, y, ...)
{
  Plot(x = x, ...)
})
setMethod("plot", signature(x = "smoothSpline", y = "smoothSpline"), function(x, y, ...)
{
  stopifnot(all())
  indices <- sort(union(s3(x, "x"), s3(y, "x")))
  get.y <- function(ss)
  {
#    ifelse(indices %in% s3(ss, "x"), s3(ss, "y"), as.numeric(NA))
    sapply(indices, function(i)
    {
    	s3(ss, "y")[which(s3(ss, "x") == i)]
    })
  }
  y.from.x <- get.y(x)
  y.from.y <- get.y(y)
  stopifnot(length(y.from.x) == length(y.from.y))
  plot(x = y.from.x, y = y.from.y, ...)
  abline(a = 0, b = 1, col = "blue")
})

setClass("Lowess", "smoothSpline")
setAs(from = "smoothSpline", to = "Lowess", function(from)
{
	lis <- as(from, "list")
	names(lis) <- "s3" # names(from)
	lo <- new("Lowess", lis)
	names(lo) <- "s3" # names(from)
	lo
})
setClass("movingAverage", "smoothSpline")
setAs(from = "smoothSpline", to = "movingAverage", function(from)
{
	movingAverage(x = from)
})
smoothData.classes <- c("Lowess", "movingAverage", "smoothSpline")
setClassUnion("smoothData", smoothData.classes)

setClass("movingLocation", representation(x = "numeric", y = "numeric"))
setMethod("lines", signature(x = "movingLocation"), function(x, ...)
{
  Lines(x = x, ...)
})
setMethod("plot", signature(x = "movingLocation", y = "missing"), function(x, y, ...)
{
  Plot(x = x, ...)
})

setClass("oneLeftOut", representation(training = "list", test = "list", noneLeftOut = "list"))
setValidity("oneLeftOut", function(object)
{
  len.ok <- length(object@training) >= 2 && length(object@training) == length(object@test) && length(object@noneLeftOut) == 1
  lens.ok <- all(length(noneLeftOut(object)) == sapply(object@test, length)) && all(length(noneLeftOut(object)) == sapply(object@training, length))
  cla.ok <- all(class(noneLeftOut(object)) == sapply(object@training, class) && class(noneLeftOut(object)) == sapply(object@test, class))
  ok <- len.ok && lens.ok && cla.ok
  if(!ok){message("invalid oneLeftOut"); browser()}
  ok
})
setMethod("sort", signature(x = "oneLeftOut", decreasing = "ANY"), function(x, decreasing = logical(0), verbose, ...)
{
	if(missing(verbose)) verbose <- FALSE
	if(missing(decreasing) || length(decreasing) == 0) decreasing <- FALSE
	nlo <- noneLeftOut(x)
	if(is(nlo, "xprnScore.p"))
	{
		sorted.olo <- sort(oneLeftOut(object = x, FUN = function(score.p)
		{
			as.xprnScore(from = score.p, caller = '"sort", signature(x = "oneLeftOut", decreasing = "ANY")')
		}))
		sorted.olo.ok <- is(sorted.olo, "oneLeftOut") && is(noneLeftOut(sorted.olo), "xprnScore")
		if(!sorted.olo.ok)
		{ message("bad sorted.olo"); browser()}
		oneLeftOut(object = x, anotherLeftOut = sorted.olo, FUN = function(score.p, sorted.score)
		{
			args.ok <- is(score.p, "xprnScore.p") && is(sorted.score, "xprnScore")
			if(!args.ok)
			{ message("list element of wrong class"); browser()}
			unsorted.p <- score.p@p
			if(is.null(names(unsorted.p)) || !sameNames(unsorted.p, sorted.score, order.sensitive = FALSE))
			{ message("name inconsistency A"); browser()}
			sorted.p <- unsorted.p[names(sorted.score)]
			sorted.p@sorted <- TRUE
			if(!sameNames(sorted.score, sorted.p, order.sensitive = TRUE))
			{ message("name inconsistency B"); browser()}
			xprnScore.p(sorted.score, p = sorted.p)
		})
	}
	else if(is(nlo, "xprnScore") || is(nlo, "estimate") || is(nlo, "sortableEstimate")) # classes defined in reprod.r and estimate.r
	{
		get.ord <- function(object)
		{
			Or <- try(Order(object = object, decreasing = decreasing, ...))
			if(is(Or, "try-error"))
			{ message("bad Or"); browser()}
			Or
		}
	  for(i in 1:length(x@test))
	  {
	    trainingSco <- x@training[[i]]
	    testSco <- x@test[[i]]
	    stopifnot(is(trainingSco, class(nlo)) && is(testSco, class(nlo)) && length(trainingSco) == length(testSco))
	  	ord <- get.ord(object = trainingSco)
	  	x@training[[i]] <- trainingSco[ord]
	  	x@test[[i]] <- testSco[ord]
	  	stopifnot(all(names(x@training[[i]]) == names(x@test[[i]])))
	  }
	  nloSco <- noneLeftOut(x)
	  ord.nlo <- get.ord(object = nloSco)
	  x@noneLeftOut <- list(nloSco[ord.nlo])
	  if(all(names(nloSco) == names(noneLeftOut(x))))
	  { message("sort failed to change order of names"); browser()}
	  else if(verbose)
			print(data.frame(names(nloSco), names(noneLeftOut(x)))[1:10, ])	  
	  x
	}
	else
#	{
#		
#	}
	{ message('"sort", signature(x = "oneLeftOut", decreasing = "ANY") not yet implemented for ', class(nlo)); browser()}
})
setMethod("plot", signature(x = "oneLeftOut", y = "missing"), function(x, y, call.sort, main, ...)
{
  nlo <- noneLeftOut(x)
  if(missing(call.sort))
  {
    call.sort <- is(nlo, "xprnScore")
    if(!is.factor(nlo))
      message("reproducibility call.sort = ", call.sort)
  }
  if(missing(main)) main <- paste("n = ", length(x@test), "; based on ", class(nlo), sep = "")
	if(is.factor(nlo) || is(nlo, "xprnScore"))
	{
		reprod <- reproducibility(x, call.sort = call.sort)
		plot(x = reprod, main = main, ...)
	}
	else
	  stop(paste('"plot", signature(x = "oneLeftOut", y = "missing") not yet implemented for ', class(noneLeftOut(x)), sep = ""))
})

setClassUnion("Vector", c("numeric", "character", "logical"))
setClass("predictionError", representation("matrix", parameter = "Vector", parameter.lab = "character")) # parameter was "numeric" before 8 October 2007
setValidity("predictionError", function(object)
{
	len.ok <- ncol(object) == length(object@parameter) && length(object@parameter.lab) <= 1
	pe.ok <- !is.na(len.ok) && len.ok
	if(!pe.ok)
	{ printInvalid(object); browser()}
	pe.ok
})
setMethod("[", signature(x = "predictionError", i = "ANY", j = "missing"), function(x, i, j, drop)
{
	rn <- rownames(x)
	x@.Data <- as(x, "matrix")[i, ]
	if(is.character(rn))
		rownames(x) <- rn[i]
	stopifnot(validObject(x))
	x
})
setMethod("plot", signature(x = "predictionError", y = "missing"), function(x, y, cumulative, indices, nfeatures, use.lower.indices, ylim, col, xlab, log, FUN, f, smooth, verbose, legend.x, ...)
{
	if(missing(legend.x)) legend.x <- "top"
	if(missing(verbose)) verbose <- FALSE
	if(verbose)
	{
		message("plotting ", class(x), " with cumulative ", if(missing(cumulative)) "[missing]" else cumulative, " and this for ...:")
		print(names(list(...)))
	}
	if(missing(log))
		log <- "y"
	if(missing(xlab))
		xlab <- "index or rank"
	if(missing(cumulative))
	{
		cumulative <- !missing(use.lower.indices)
		message("predictionError cumulative = ", cumulative)
	}
	if(missing(indices))
	{
		if(missing(nfeatures) || length(nfeatures) == 0)
		{
			nfeatures <- if(cumulative) min(2000, nrow(x)) else nrow(x)
			message("nfeatures = ", nfeatures)
		}
		if(missing(use.lower.indices))
		{
			use.lower.indices <- if(cumulative) TRUE else logical(0)
			message("use.lower.indices = ", use.lower.indices)
		}
		nf <- ceiling(min(nfeatures, if(length(use.lower.indices) == 0) nrow(x) / 2 else nrow(x)))
		lower <- 1:nf
		upper <- (nrow(x) + 1 - nf):nrow(x)
		indices <- if(length(use.lower.indices) == 0)
			union(lower, upper)
		else if(use.lower.indices)
			lower
		else
			upper
	}
	if(missing(smooth))
	{
		smooth <- if(cumulative)
			FALSE
		else
		{
			if(missing(f))
			{
				f <- 1/100
				if(missing(smooth) || !identical(smooth, FALSE))
					message("f = ", f)
			}
			stopifnot(is.numeric(f) && length(f) == 1)
			smoothDataFUN(f = f)
		}
	}
	smooth.x <- function(vec)
	{
		stopifnot(is.numeric(vec))
		if(is.logical(smooth) && !smooth)
			vec # cf. plot biasEstimates
		else
			smooth(vec)
	}
	x <- smooth.x(x)
	if(missing(FUN))
	{
		FUN.name <- if(cumulative) "identity" else "exp.root"
		message(class(x), " FUN = ", FUN.name)
		FUN <- eval(parse(text = FUN.name))
	}
	stopifnot(is.function(FUN))
	sub.y.mat <- if(cumulative)
	{
		if(verbose) message("calling accumulate with indices from ", min(indices), " to ", max(indices), ".")
		accumulate(x, indices = indices)
	}
	else
		x[indices, ]
	if(missing(ylim))
		ylim <- FUN(range(as.numeric(sub.y.mat), na.rm = TRUE))
	graph <- function(fun, indices, sub.y, ...)
	{
		stopifnot(length(sub.y) <= nrow(x))
#		sub.y <- get.sub.y(indices = indices, y = y)
		stopifnot(length(sub.y) == length(indices))
		fun(x = indices, y = FUN(sub.y), ...)
	}
	pch <- 1:length(x@parameter)
	if(missing(col))
		col <- pch
	if(length(col) == 1)
		col <- rep(col, length(x@parameter))
	stopifnot(length(col) == length(pch))
	arglis <- list(...)
	for(j in 1:length(x@parameter))
	{
		param <- x@parameter[j]
		fun <- if(j == 1)
			function(...)
			{
				do.call("plot", c(arglis, list(xlab = xlab, ylab = "prediction error", log = log, ylim = ylim, ...)))
			}
		else
			points
		graph(fun = fun, indices = indices, sub.y = sub.y.mat[, j], col = col[j], pch = pch[j])
	}
	legend(legend = if(length(x@parameter.lab) == 1) paste(x@parameter.lab, x@parameter, sep = " = ") else x@parameter, x = legend.x, col = col, pch = pch)
})

# setClass("general.numeric")
# setValidity("general.numeric", function(object)
# {
# 	ok <- is(object, "numeric")
# 	if(!ok)
# 	{ printInvalid(object); browser()}
# 	ok
# })

if(!isClass("numericEstimate"))
	setClassUnion("numericEstimate", c("numeric")) # modified in the file "estimate.r"

setClass("biasEstimate", representation("numeric", uncorrected = "numericEstimate", sorted = "logical", jackknife = "logical", Rank = "Numeric", Weight = "Scalar")) # changed 16 April 2008
setValidity("biasEstimate", function(object)
{
	ra <- object@Rank
	ra.ok <- length(ra) == 0 || (object@sorted && length(ra) == length(object) && all(names(ra) == names(object)) && all(ra >= 1))
	cla.ok <- is(object@uncorrected, "numeric")
	is.boo.ok <- function(boo){length(boo) == 1}
	boo.ok <- is.boo.ok(object@sorted) && is.boo.ok(object@jackknife)
	len.ok <- length(object) == length(object@uncorrected)
	nam.ok <- all(names(object) == names(object@uncorrected))
	ok <- ra.ok && cla.ok && boo.ok && !is.na(nam.ok) && nam.ok
	if(!ok)
	{ printInvalid(object); browser()}
	ok
})
setMethod("names", signature(x = "biasEstimate"), function(x){names(x@uncorrected)})
setMethod("annotation", signature(object = "biasEstimate"), function(object)
{
	ann <- try(annotation(object@uncorrected), silent = TRUE)
	if(is(ann, "try-error"))
		"uncorrected estimate"
	else
		ann
})
setAs(from = "biasEstimate", to = "numeric", function(from)
{
	vec <- as.numeric(from)
	names(vec) <- names(from)
	vec
})
setMethod("[", signature(x = "biasEstimate", i = "ANY", j = "missing"), function(x, i, j, drop)
{
  bias <- as(x, "numeric")[i]
  uncorrected <- x@uncorrected[i]
  Ra <- if(x@sorted)
  	Rank(x)[i]
  else
  	x@Rank # Numeric(numeric(0))
  new.biasEstimate(bias = bias, uncorrected = uncorrected, sorted = x@sorted, jackknife = x@jackknife, Rank = Ra)
})
setMethod("plot", signature(x = "biasEstimate", y = "missing"), function(x, y, name, sub, xlab, ...)
{
	if(missing(name))
	{
		get.type <- function() # requires fdr.s code
		{
			x@uncorrected@estimator@type
		}
		type <- try(get.type(), silent = TRUE)
		name <- if(is(type, "try-error"))
			"uncorrected"
		else
			type
		message("name = ", name)
	}
	new.x <- if(name == "pvalue.z")
		qvalue(x)
	else if(name %in% slotNames(x))
		slot(x, name = name)
	else
		stop('bad name in "plot", signature(x = "biasEstimate", y = "missing")')
	if(missing(sub))
	{
		sub <- if(name == "pvalue.z")
			""
		else
			name
	}
	if(missing(xlab))
	{
		xlab <- if(name == "pvalue.z")
			"q-value"
		else
			annotation(x)
	}
	plot(x = new.x, y = x, xlab = xlab, ylab = "estimate", sub = sub, ...)
})
setMethod("plot", signature(x = "numeric", y = "biasEstimate"), function(x, y, xlab, ylab, smooth, call.browser, save.space, f, include.estimate, main, call.par, include.rough, ...)
{
	if(!y@sorted)
	{
		sort.y <- function()
		{
			message("sorting ", class(y), " on ", date())
			sorted.y <- sort(y)
			if(!sorted.y@sorted)
			{ message("bad sorted.y"); browser()}
			sorted.y
		}
		y <- sort.y()
	}
	if(missing(include.estimate))
	{
		include.estimate <- TRUE
		message("include.estimate = ", include.estimate)
	}
	if(missing(include.rough))
	{
		include.rough <- !include.estimate
		message("include.rough = ", include.rough)
	}
	if(missing(save.space))
	{
		save.space <- TRUE
		message("save.space = ", save.space)
	}
	stopifnot(length(x) == length(y) && is.character(names(x)) && is.character(names(y)))
	if(missing(call.par))
		call.par <- TRUE
	same.names <- function()
	{
		length(x) == length(y) && is.character(names(x)) && is.character(names(y)) && all(names(x) == names(y))
	}
	if(!same.names())
		x <- x[names(y)]
	stopifnot(same.names())
	if(missing(call.browser)) call.browser <- FALSE
	if(call.browser) browser()
	if(missing(smooth))
		smooth <- NULL
	if(missing(f))
		f <- NULL
	if(include.rough)
		y.rough <- y
	y <- Smooth(object = y, smooth = smooth, f = f)
	if(include.estimate && call.par) par(mfrow = c(2, 2))
	uncorrected.pch <- 1
	corrected.pch <- 2
	uncorrected.col <- "orange"
	corrected.col <- "blue"
	if(missing(main))
		main <- annotation(y)
	graph <- function(FUN, y.vec, ...)
	{
		FUN(x = as.numeric(x), y = as.numeric(y.vec), ...)
	}
	if(missing(ylab))
		ylab <- "estimate"
	if(include.estimate)
	{
		graph(FUN = plot, y.vec = y@uncorrected, xlab = xlab, ylab = ylab, pch = uncorrected.pch, col = uncorrected.col, main = main)
		graph(FUN = points, y.vec = corrected(y), pch = corrected.pch, col = corrected.col)
		if(!save.space)
			blank.plot()
		legend(x = if(save.space) "right" else "left", legend = c("uncorrected estimate", "corrected estimate"), pch = c(uncorrected.pch, corrected.pch), col = c(uncorrected.col, corrected.col), bg = "white")
	}
	rough.col <- "black"
	graph(FUN = plot, y.vec = if(include.rough) y.rough else y, xlab = xlab, ylab = "estimated bias", main = main, col = if(include.rough) rough.col else "black", ...)
	if(include.rough)
	{
		if(all(y.rough == y, na.rm = TRUE))
			warning("unsmoothed points do not differ from smoothed points")
		graph(FUN = points, y.vec = y, type = "l", col = "orange", lwd = 2)
	}
})
setMethod("plot", signature(x = "biasEstimate", y = "biasEstimate"), function(x, y, call.par, ...)
{
	if(missing(call.par))
		call.par <- TRUE
	if(call.par)
		par(mfrow = c(2, 2))
	xlab <- annotation(x)
	ylab <- annotation(y)
	graph <- function(FUN, main)
	{
		get.vec <- function(object){as.numeric(FUN(object))}
		plot(x = get.vec(x), y = get.vec(y), xlab = xlab, ylab = ylab, main = main, ...)
		abline(a = 0, b = 1, col = "blue")
	}
	graph(FUN = function(object){object@uncorrected}, main = "uncorrected estimates")
	graph(FUN = corrected, main = "corrected estimates")
	graph(FUN = identity, main = "bias")
})

setClass("Density", representation(s3 = "list", annotation = "character"))
setValidity("Density", function(object)
{
	s3.ok <- length(object@s3) == 1 && names(object@s3) == "s3"
	ann.ok <- length(object@annotation %in% c(0, 1))
	ok <- s3.ok && ann.ok
	if(!ok)
	{ printInvalid(object); browser()}
	ok
})
setMethod("plot", signature(x = "Density", y = "missing"), function(x, y, main, ...)
{
	if(missing(main)) main <- annotation(x)
	plot(s3(x), main = main, ...)
})
setAs(from = "Density", to = "function", function(from)
{
	den <- s3(from)
	dom <- domain(from)
	function(v)
	{
		adens <- approxfun(x = den$x, y = den$y)(v)
		stopifnot(length(v) == length(adens))
		dens <- try(ifelse(v >= dom[1] & v <= dom[2], adens, 0))
		if(is(dens, "try-error"))
		{ message("bad dens"); browser()}
		dens
	}
})

setClass("functions", representation("list"))
setValidity("functions", function(object)
{
	length(object) == 0 || all(sapply(object, is, class2 = "function"))
})

setMethod("plot", signature(x = "list", y = "missing"), function(x, y, call.par, ...) # based on vplot3
{
	if(missing(call.par))
		call.par <- length(x) > 1
	if(call.par)
		par(mfrow=c(2,2))
	for(i in 1:length (x))
	{
		message("plotting ", class (x[[i]]))
		plot(x=x[[i]], ...)
	}
})

setAs(from = "ANY", to = "list", function(from)
{
	nam <- slotNames(from)
	lis <- lapply(nam, function(name){slot(from, name = name)})
	names(lis) <- nam
	lis
})




# near end of file:

Source(file = "data.s") # functions
