# BiobaseSupp, Copyright (c) 2002-2004 by David R. Bickel.
# Last modified by David R. Bickel on 3/25/04.
# Created by David R. Bickel on 6/13/02.
# These functions are mostly wrappers of Statomics functions for use with the Biobase package of Bioconductor.org.


# 2/1/04. __? changed to max.null.auc.
# 6/20/03. The use of make.names was modified and expanded in default.read.exprs and default.read.pData.
# 8/20/02. unpaired... changed to boot...; using.p.value.stats added to bioDiffTests, bioPairedDiffTests, and bioGenPairedDiffTests.
# 8/7/02.  paired tests now allow one-sided alternatives.
# 8/5/02. Made to depend on MultTests.r instead of DifferenceTests.r.

source("MultTests.r")
source("MultCorr.r")
library(Biobase) # from bioconductor.org

default.read.exprs <- function(exprs.file)
{
  exprs.table <- read.delim(exprs.file, as.is = TRUE, comment.char = '')
  names(exprs.table) <- make.names(names(exprs.table), unique = TRUE)
  row.names(exprs.table) <- make.names(exprs.table[, 1], unique = TRUE)
  as.matrix(exprs.table[, 2:ncol(exprs.table)])
}

default.read.pData <- function(pData.file)
{
  pd <- read.delim(pData.file, comment.char = '')
  if(!identical(TRUE, all.equal(pd[, 1], unique(pd[, 1])))) {cat('The names of',pData.file,'should be unique, but the names of the following rows are duplicates:\n'); print(pd[duplicated(pd[, 1]), ]); stop('Duplicated name.')}
  pdata <- read.delim(pData.file, row.names=1, comment.char = '')
  pdata2 <- data.frame(apply(pdata, MARGIN = 2, FUN = function(column){make.names(column, unique = FALSE)}))
  rownames(pdata2) <- make.names(rownames(pdata), unique = TRUE)
  new('phenoData', pData = pdata2, varLabels = as.list(make.names(names(pdata2), unique = TRUE)))
}

read.exprSet <- function(exprSet.name, read.exprs = default.read.exprs, read.pData = default.read.pData, exprs.file = paste(exprSet.name, 'exprs.txt', sep='-'), pData.file = paste(exprSet.name, 'pData.txt', sep='-'))
{
  phenoData <- read.pData(pData.file)
  exprs <- read.exprs(exprs.file)
  exprSet <- new('exprSet', exprs = exprs, phenoData = phenoData)
  if(ncol(exprs(exprSet)) != nrow(pData(exprSet)))
  {
    print(ncol(exprs(exprSet)))
    print(nrow(pData(exprSet)))
    stop(paste(exprs.file, 'conflicts with', pData.file))
  }
  exprs.samples <- sampleNames(exprSet)
  phenoData.samples <- row.names(pData(exprSet))
  if(!identical(TRUE, all.equal(exprs.samples, phenoData.samples)))
    warning('Inconsistent sample names: exprs.samples != phenoData.samples\n', paste(exprs.samples, collapse=', '), '\n', paste(phenoData.samples, collapse=', '))
  exprSet
}

default.normalization.fcn <- function(sample.expr, scale.factor.fcn = median, translation.fcn = function(x) {0})
{
  norm <- scale.factor.fcn(sample.expr)
  if(norm <= 0)
    stop('Scale factor is ', 1/norm, '\n')
  (sample.expr - translation.fcn(sample.expr)) / norm
}

replace.exprs <- function(exprSet, FUN)
{
  new('exprSet', exprs = FUN(exprs(exprSet)), phenoData = phenoData(exprSet))
}

normalize.microarrays <- function(exprSet, normalization.fcn = default.normalization.fcn)
{
#  exprs(exprSet) <- apply(exprs(exprSet), MARGIN = 2, normalization.fcn)
  replace.exprs(exprSet, function(exprs) {apply(exprs, MARGIN = 2, normalization.fcn)})
}

pos.trans <- function(x, power=1, prefactor=1) {ifelse(x>0, {y<-x^power; log(1+prefactor*y)/prefactor}, x)}

real.trans <- function(x) {abs.real.trans <- pos.trans(abs(x)); ifelse(x >= 0, abs.real.trans, -abs.real.trans)}

transform.exprs <- function(exprSet, transform = real.trans) {replace.exprs(exprSet, transform)}

preprocess.exprs <- function(exprSet, normalization.fcn = default.normalization.fcn, transform = real.trans)
{
  transform.exprs(normalize.microarrays(exprSet, normalization.fcn = normalization.fcn), transform = transform)
}

rescale.exprs <- function(exprSet, isInGroupVecByGroup, scale.estimator = sd)
{
  stopifnot(is.matrix(isInGroupVecByGroup))
  trues <- apply(isInGroupVecByGroup, 1, sum)
  stopifnot(length(trues) == nrow(isInGroupVecByGroup))
  if(any(trues > 1)) stop('Overlapping groups are not allowed.')
  if(any(trues < 1)) warning('Some of the data is not used in rescaling.')
  
  scales <- apply(exprs(exprSet), MARGIN = 1, FUN = function(gene) {pooled.scale(x = lapply(1:ncol(isInGroupVecByGroup), function(i) {gene[isInGroupVecByGroup[, i]]}), scale.estimator = scale.estimator)})
  stopifnot(length(scales) == nrow(exprs(exprSet)))
  
  replace.exprs(exprSet, FUN = function(exprs) {exprs / scales})
}



#exprSet.as.matrix <- function(exprSet)
#{
#  exprs <- exprs(exprSet)
#  pData <- pData(exprSet)
#  matrix(c(t(exprs), as.matrix(pData)), nrow = ncol(exprs), ncol = nrow(exprs) + ncol(pData))
#}

exprSet.as.data.frame <- function(exprSet, pVariables = names(pData(exprSet)))
{
  exprs <- exprs(exprSet)
  pData <- pData(exprSet)
  if(!all(sapply(pVariables, function(pVar){any(pVar == names(pData))}))) stop(cat('ERROR: ', pVariables, 'are not in exprSet'))
  datf <- do.call('data.frame', c(list(t(exprs)), lapply(pVariables, function(pVar){pData[, pVar]})))
  names(datf) <- make.names(c(geneNames(exprSet), pVariables), unique = TRUE)
  datf
}

exprSet.as.sorted.data.frame <- function(exprSet, pVariables = names(pData(exprSet)))
{
  datf <- exprSet.as.data.frame(exprSet, pVariables = pVariables)
  # num.genes <- ncol(datf) - length(pVariables)
  row.order <- do.call('order', args = lapply(pVariables, function(pVar){datf[, pVar]}))
  datf[row.order, ] # The intermediate variable row.order was introduced on 7/11/02 to avoid error ["unused argument(s) (value ...)"] in Darwin and Windows versions of R 1.5.1, but now Darwin gives a "segmentation fault" error.
}

# sorted.exprs <- function(exprSet, pVariables = names(pData(exprSet)))


exprs.diff.matrix <- function(exprSet, bilevel.factor, factor.of.pairs)
# e.g., bilevel.factor = names(pData(exprSet))[1], factor.of.pairs = names(pData(exprSet))[2]
{
  pData <- pData(exprSet)
  pData <- pData[, sapply(names(pData), function(name) {name %in% c(bilevel.factor, factor.of.pairs)})]
  stopifnot(ncol(pData) == 2)
  if(!identical(dim(pData), dim(unique(pData))))
  {
    print(bilevel.factor)
    print(factor.of.pairs)
    stop('bilevel.factor and factor.of.pairs incorrectly define the pairing.')
  }
  if(!is.factor(pData[, bilevel.factor]) || length(levels(pData[, bilevel.factor])) != 2)
    stop(paste(levels(pData[, bilevel.factor]), 'should have two levels.'))
  num.pairs <- length(pData[, bilevel.factor][pData[, bilevel.factor] == pData[1, bilevel.factor]])
  if(num.pairs != nrow(pData)/2)
    stop(paste(num.pairs, nrow(pData)/2, 'Not paired.'))
  tab <- table(pData[, bilevel.factor], pData[, factor.of.pairs])
  if(!(identical(do.call('all.equal', args = c(list(as.integer(1)), as.list(tab))), TRUE)))
  {
    print(tab)
    print(bilevel.factor)
    print(factor.of.pairs)
    stop(paste('bilevel.factor and factor.of.pairs do not sufficiently specify the pairing.'))
  }
  
  exprSet2 <- exprSet
  pData(exprSet2) <- pData
  datf <- exprSet.as.sorted.data.frame(exprSet2, pVariables = c(bilevel.factor, factor.of.pairs))
  datf1 <- datf[1:(nrow(datf)/2), ]
  datf2 <- datf[-(1:nrow(datf1)), ]
  row.names(datf1) <- row.names(datf2) <- (1:length(row.names(datf1)))
  stopifnot(identical(datf1[, factor.of.pairs], datf2[, factor.of.pairs]))
  
  num.genes <- nrow(exprs(exprSet))
  mat <- t(datf1[, 1:num.genes] - datf2[, 1:num.genes])
  rownames(mat) <- geneNames(exprSet)
  colnames(mat) <- datf1[, factor.of.pairs]
  stopifnot(colnames(mat) == unique(colnames(mat)))
  mat
}

exprs.null.diff.matrix <- function(null.exprSets, null.bilevel.factors, null.factors.of.pairs)
# null.exprSets is a list, each element of which is an exprSet, and null.bilevel.factors is a character vector.
{
  stopifnot(length(null.bilevel.factors) == length(null.exprSets)) == length(null.factors.of.pairs)
  exprSet.factors.list <- list()
  for(i in 1:length(null.exprSets))
    exprSet.factors.list[[i]] <-
      list(null.exprSets[[i]], null.bilevel.factors[[i]], null.factors.of.pairs[[i]])
  mats <- lapply(exprSet.factors.list,
    function(efl) {exprs.diff.matrix(efl[[1]], bilevel.factor = efl[[2]], factor.of.pairs = efl[[3]])})

  num.genes.list <- lapply(null.exprSets, function(es) {nrow(exprs(es))})
  stopifnot(length(num.genes.list) == 1 || identical(do.call('all.equal', args = num.genes.list), TRUE))

  genes.list <- lapply(null.exprSets, function(es) {geneNames(es)})
  stopifnot(length(genes.list) == 1 || identical(do.call('all.equal', args = genes.list), TRUE))

  matrix(do.call('c', args = mats), nrow = num.genes.list[[1]], dimnames = list(genes.list[[1]], do.call('c', lapply(mats, colnames))))
}

bio.boot.diff.tests <- function(exprSet, nresample, bilevel.factor = names(pData(exprSet))[1], test.name = 'bio.boot.diff.tests', file.base = test.name, location.estimator = mean, diff.scale.estimator = function(x, y) {pooled.scale(x, y, scale.estimator = meanad)}, alternative = c('two.sided', 'less', 'greater'), ...)
{
  alternative <- match.arg(alternative)
  
  errRates(nresample = nresample, file.base = file.base, multTestsObjs = list(bioBootDiffTests(exprSet = exprSet, bilevel.factor = bilevel.factor, alternative = alternative, location.estimator = location.estimator, diff.scale.estimator = diff.scale.estimator, test.name = test.name, ...)))
}

bioBootDiffTests <- function(exprSet, bilevel.factor = names(pData(exprSet))[1], location.estimator = mean, diff.scale.estimator = function(x, y) {pooled.scale(x, y, scale.estimator = meanad)}, alternative = c('two.sided', 'less', 'greater'), test.name = 'bioBootDiffTests', ...)
{
  alternative <- match.arg(alternative)
  
  bootDiffTests(two.samples = exprSet.as.data.frame(exprSet, pVariables = bilevel.factor), alternative = alternative, location.estimator = location.estimator, diff.scale.estimator = diff.scale.estimator, test.name = test.name, ...)
}

bilevel.factor.levels <- function(exprSet, bilevel.factor)
{
  vec <- sort(unique(pData(exprSet)[, bilevel.factor]))
  stopifnot(length(vec) == 2)
  if(!all(vec == levels(bilevel.factor))) warning('levels is not the same as what will be treated as the levels')
 # if(!is.vector(vec)) {print('vec should be a vector'); browser()}
  if(!all(sapply(vec, is.character))) {print('vec should have character elements only'); browser()}
  vec
}

default.subsample.sizes <- function(exprSet, bilevel.factor)
{
  levs <- bilevel.factor.levels(exprSet, bilevel.factor)
  stopifnot(all(levs == make.names(levs, unique = TRUE)))
  eval(eval.arg('c(', levs[1], ' = NA, ', levs[2], ' = NA)'))
}

# lapply(exprSets, function(exprSet){bio.wilcox.diff.tests(exprSet, ...)} could take too much RAM for a large number of bootstrapped exprSets; what is needed is a function that will create the bootstrap samples and discard them for their results. BUT you do not want to resample entire exprSets; you want to resample matrices, which would be easiest inside diffTests, so implement it for bio.wilcox.diff.tests first, and only for other functions later, as needed. Two options: 1. have diffTests create a single bootstrap sample, for use with multiple calls to bio.wilcox.diff.tests; 2. have diffTests return a list of multTests objects, causing major changes in other functions. The former seems easier, less liable to error, and possibly more in line with the goal.



bio.boot.wilcox.diff.tests <- function(nboot, summarize.err.rates = default.summarize.err.rates, ...)
# summarize.err.rates is a function of an err.rates.out object (the value returned by bio.wilcox.diff.tests) that returns a vector of named summary statistics.
# bio.boot.wilcox.diff.tests returns a (summary statistic) by (bootstrap sample) matrix of summary statistics for all bootstrap samples.
{
  resample.tests(nresample = nboot, summarize.err.rates = summarize.err.rates, tests.fun = function(b){bio.wilcox.diff.tests(bootstrap.sample = TRUE, ...)})
}

bio.subsample.wilcox.diff.tests <- function(nsubsample, summarize.err.rates = default.summarize.err.rates, exprSet, bilevel.factor = names(pData(exprSet))[1], subsample.sizes, file.base = 'bio.subsample.wilcox.diff.tests', fdr = 0.2, ...)
{
  stopifnot(all(is.finite(subsample.sizes)))
  m.n.0 <- two.sample.sizes(exprSet.as.data.frame(exprSet, pVariables = bilevel.factor))
#  if(is.null(names(m.n.0))) {print('m.n.0 lost its names'); browser()}
#  m.n.sub <- sapply(names(m.n.0), function(name){subsample.sizes[name]})
  m.n.sub <- subsample.sizes

  resample.tests(nresample = nsubsample, summarize.err.rates = summarize.err.rates, tests.fun = function(b){adjust.wilcox.err.rates(bio.wilcox.diff.tests(exprSet = exprSet, bilevel.factor = bilevel.factor, subsample.sizes = subsample.sizes, fdr = 0, file.base = '', show.progress = (b == 1), ...), old.m = m.n.sub[1], old.n = m.n.sub[2], new.m = m.n.0[1], new.n = m.n.0[2], file.base = if(b == 1) file.base else '', using.p.value.stats = FALSE, show.progress = (b == 1), fdr = fdr)})
}

bio.wilcox.diff.tests <- function(exprSet, bilevel.factor = names(pData(exprSet))[1], subsample.sizes = default.subsample.sizes(exprSet, bilevel.factor), bootstrap.sample = FALSE, test.name = 'bio.wilcox.diff.tests', file.base = test.name, max.null.auc = 0.5, alternative = c('two.sided', 'less', 'greater'), balance = FALSE, desum = FALSE, regress.file = '', nboot.rescale = 0, diff.scale.estimator = 'none', scale.file = '', show.progress = TRUE, fdr = 0.2, ...)
{
  alternative <- match.arg(alternative)
  stopifnot(alternative == 'two.sided')
  two.samples <- exprSet.as.data.frame(exprSet, pVariables = bilevel.factor)
  effect.size <- null.zeroed.wilcox.statistic(max.null.auc = max.null.auc, two.samples = two.samples)
  
  h0DistErrRates(null.cdfs = list(function(stat){wilcox.null.cdf(stat, two.samples = two.samples)}), effect.sizes = effect.size, file.base = file.base, show.progress = show.progress, null.quant.funs = list(function(p){wilcox.null.quant.fun(p, two.samples = two.samples)}), fdrs = fdr, multTestsObjs = list(bioDiffTests(exprSet = exprSet, bilevel.factor = bilevel.factor, subsample.sizes = subsample.sizes, bootstrap.sample = bootstrap.sample, alternative = alternative, test.stat = wilcox.test.stat, balance = balance, desum = desum, regress.file = regress.file, nboot.rescale = nboot.rescale, diff.scale.estimator = diff.scale.estimator, scale.file = scale.file, test.name = test.name, show.progress = show.progress, ...)))
}

bio.t.diff.tests <- function(exprSet, bilevel.factor = names(pData(exprSet))[1], subsample.sizes = default.subsample.sizes(exprSet, bilevel.factor), test.name = 'bio.t.diff.tests', file.base = test.name, alternative = c('two.sided', 'less', 'greater'), balance = FALSE, desum = FALSE, regress.file = '', nboot.rescale = 0, diff.scale.estimator = 'none', scale.file = '', show.progress = TRUE, fdr = 0.2, ...)
{
  alternative <- match.arg(alternative)
  stopifnot(alternative == 'two.sided')
  two.samples <- exprSet.as.data.frame(exprSet, pVariables = bilevel.factor)
  
  h0DistErrRates(null.cdfs = list(function(stat){t.null.cdf(stat, two.samples = two.samples)}), file.base = file.base, show.progress = show.progress, null.quant.funs = list(function(p){t.null.quant.fun(p, two.samples = two.samples)}), fdrs = fdr, multTestsObjs = list(bioDiffTests(exprSet = exprSet, bilevel.factor = bilevel.factor, subsample.sizes = subsample.sizes, alternative = alternative, test.stat = t.test.stat, balance = balance, desum = desum, regress.file = regress.file, nboot.rescale = nboot.rescale, diff.scale.estimator = diff.scale.estimator, scale.file = scale.file, test.name = test.name, show.progress = show.progress, ...)))
}

bio.h0dist.diff.tests <- function(exprSet, null.cdf, bilevel.factor = names(pData(exprSet))[1], test.name = 'bio.h0dist.diff.tests', file.base = test.name, alternative = c('two.sided', 'less', 'greater'), test.stat = gen.t.stat, balance = FALSE, desum = FALSE, regress.file = '', nboot.rescale = 0, diff.scale.estimator = 'none', scale.file = '', show.progress = TRUE, null.quant.fun = NULL, fdr = if(is.null(quant.null.fun)) NULL else 0.2, ...)
{
  alternative <- match.arg(alternative)
  
  h0DistErrRates(null.cdfs = list(null.cdf), file.base = file.base, null.quant.funs = if(is.null(null.quant.fun)) NULL else list(null.quant.fun), fdrs = fdr, multTestsObjs = list(bioDiffTests(exprSet = exprSet, bilevel.factor = bilevel.factor, alternative = alternative, test.stat = test.stat, balance = balance, desum = desum, regress.file = regress.file, nboot.rescale = nboot.rescale, diff.scale.estimator = diff.scale.estimator, scale.file = scale.file, test.name = test.name, show.progress = show.progress, ...)))
}

bio.diff.tests <- function(exprSet, nresample, bilevel.factor = names(pData(exprSet))[1], test.name = 'bio.diff.tests', file.base = test.name, alternative = c('two.sided', 'less', 'greater'), test.stat = gen.t.stat, balance = FALSE, desum = FALSE, regress.file = '', nboot.rescale = 0, diff.scale.estimator = 'none', scale.file = '', input.file.base = '', show.progress = TRUE, ...)
{
  alternative <- match.arg(alternative)
  
  errRates(nresample = nresample, file.base = file.base, input.file.base = input.file.base, multTestsObjs = list(bioDiffTests(exprSet = exprSet, bilevel.factor = bilevel.factor, alternative = alternative, test.stat = test.stat, balance = balance, desum = desum, regress.file = regress.file, nboot.rescale = nboot.rescale, diff.scale.estimator = diff.scale.estimator, scale.file = scale.file, test.name = test.name, show.progress = show.progress, ...)))
}

bioDiffTests <- function(exprSet, bilevel.factor = names(pData(exprSet))[1], subsample.sizes = default.subsample.sizes(exprSet, bilevel.factor), bootstrap.sample = FALSE, alternative = c('two.sided', 'less', 'greater'), test.stat = gen.t.stat, using.p.value.stats = identical(test.stat, gen.t.pstat), balance = FALSE, desum = FALSE, regress.file = '', nboot.rescale = 0, diff.scale.estimator = 'none', scale.file = '', test.name = 'bioDiffTests', show.progress = TRUE, ...)
{
  alternative <- match.arg(alternative)
  
  diffTests(two.samples = exprSet.as.data.frame(exprSet, pVariables = bilevel.factor), subsample.sizes = subsample.sizes, bootstrap.sample = bootstrap.sample, alternative = alternative, test.stat = test.stat, using.p.value.stats = using.p.value.stats, balance = balance, desum = desum, regress.file = regress.file, nboot.rescale = nboot.rescale, diff.scale.estimator = diff.scale.estimator, scale.file = scale.file, test.name = test.name, show.progress = show.progress, ...)
}

bio.effect.size.estimation <- function(exprSet, bilevel.factor, nresample, alternative = c('two.sided', 'less', 'greater'), test.stat = auc.test.stat, subsample = TRUE, file = 'effect.size.xls', using.p.value.stats = FALSE, browse = FALSE, jitter.sd = NULL)
{
  alternative <- match.arg(alternative)
  
  effect.size.estimation(two.samples = exprSet.as.data.frame(exprSet, pVariables = bilevel.factor), nresample = nresample, alternative = alternative, test.stat = test.stat, subsample = subsample, file = file, using.p.value.stats = using.p.value.stats, browse = browse, jitter.sd = jitter.sd)
}

bio.paired.diff.tests <- function(exprSet, nresample, bilevel.factor = names(pData(exprSet))[1], test.name = 'bio.paired.diff.tests', file.base = test.name, factor.of.pairs = names(pData(exprSet))[2], randomize.signs = sign.permute.columns, alternative = c('two.sided', 'less', 'greater'), test.stat = gen.paired.t.stat, monitor.test.stat = NULL, ...)
{
  alternative <- match.arg(alternative)
  
  errRates(nresample = nresample, file.base = file.base, multTestsObjs = list(bioPairedDiffTests(exprSet = exprSet, bilevel.factor = bilevel.factor, factor.of.pairs = factor.of.pairs, alternative = alternative, randomize.signs = randomize.signs, test.stat = test.stat, monitor.test.stat = monitor.test.stat, test.name = test.name, ...)))
}

bioPairedDiffTests <- function(exprSet, bilevel.factor = names(pData(exprSet))[1], factor.of.pairs = names(pData(exprSet))[2], randomize.signs = sign.permute.columns, alternative = c('two.sided', 'less', 'greater'), test.stat = gen.paired.t.stat, using.p.value.stats = identical(test.stat, gen.paired.t.pstat), monitor.test.stat = NULL, test.name = 'bioPairedDiffTests', ...)
{
  alternative <- match.arg(alternative)
  
  pData <- pData(exprSet)
  if(length(unique(pData[, bilevel.factor])) != 2) stop(paste(pData[, bilevel.factor]), 'should have 2 unique elements!')
  if(length(unique(pData[, factor.of.pairs])) * 2 != length(pData[, factor.of.pairs])) stop(paste(pData[, factor.of.pairs],'has the wrong number of unique elements for factor.of.pairs'))

  pairedDiffTests(d2 = exprSet.as.data.frame(exprSet = exprSet, pVariables = c(bilevel.factor, factor.of.pairs)), alternative = alternative, randomize.signs = randomize.signs, test.stat = test.stat, using.p.value.stats = using.p.value.stats, monitor.test.stat = monitor.test.stat, test.name = test.name, ...)
}

bio.gen.paired.diff.tests <- function(exprSet, nresample, bilevel.factor = names(pData(exprSet))[1], factor.of.pairs, null.exprSets, null.bilevel.factors, null.factors.of.pairs = rep(factor.of.pairs, length(null.exprSets)), test.name = 'bio.gen.paired.diff.tests', file.base = test.name, sample.within.levels = (length(null.exprSets) > 1), randomize.signs = sign.permute.columns, alternative = c('two.sided', 'less', 'greater'), test.stat = gen.paired.t.stat, monitor.test.stat = NULL, ...)
{
  alternative <- match.arg(alternative)
  
  errRates(nresample = nresample, file.base = file.base, multTestsObjs = list(bioGenPairedDiffTests(exprSet = exprSet, bilevel.factor = bilevel.factor, alternative = alternative, factor.of.pairs = factor.of.pairs, null.exprSets = null.exprSets, null.bilevel.factors = null.bilevel.factors, null.factors.of.pairs = null.factors.of.pairs, sample.within.levels = sample.within.levels, randomize.signs = randomize.signs, test.stat = test.stat, monitor.test.stat = monitor.test.stat, test.name = test.name, ...)))
}

bioGenPairedDiffTests <- function(exprSet, bilevel.factor, factor.of.pairs, null.exprSets, null.bilevel.factors, null.factors.of.pairs = rep(factor.of.pairs, length(null.exprSets)), sample.within.levels = (length(null.exprSets) > 1), randomize.signs = sign.permute.columns, alternative = c('two.sided', 'less', 'greater'), test.stat = gen.paired.t.stat, using.p.value.stats = identical(test.stat, gen.paired.t.pstat), monitor.test.stat = NULL, test.name = 'bioGenPairedDiffTests', ...)
{
  alternative <- match.arg(alternative)
  
  genPairedDiffTests(diffs = exprs.diff.matrix(exprSet = exprSet, bilevel.factor = bilevel.factor, factor.of.pairs = factor.of.pairs), null.diffs = exprs.null.diff.matrix(null.exprSets = null.exprSets, null.bilevel.factors = null.bilevel.factors, null.factors.of.pairs = null.factors.of.pairs), alternative = alternative, levels.compared = levels(pData(exprSet)[, bilevel.factor]), sample.within.levels = sample.within.levels, randomize.signs = randomize.signs, test.stat = test.stat, using.p.value.stats = using.p.value.stats, monitor.test.stat = monitor.test.stat, test.name = test.name, ...)
}

bio.new.correlation.table <- function(exprSet, permute.columns = FALSE, permute.row.wise = FALSE, transform.fun = rank, max.num.clusters = 0, cluster.alpha = 2, two.sided = TRUE, correlation.threshold = 0.9, corromics.java.path = default.corromics.java.path, file = 'bio.corr.table.xls', view.table = (!is.null(file) && file != ''), corromics.log.file = 'corromics.log.txt')
{
  new.correlation.table(vectorRows = exprs(exprSet), permute.columns = permute.columns, permute.row.wise = permute.row.wise, transform.fun = transform.fun, max.num.clusters = max.num.clusters, cluster.alpha = cluster.alpha, two.sided = two.sided, correlation.threshold = correlation.threshold, corromics.java.path = corromics.java.path, file = file, corromics.log.file = corromics.log.file)
}
