# Proteomics, Copyright (c) 2003 by David R. Bickel.

source("MultTests.r")

is.prot.frame <- function(prot.frame)
{
  is.data.frame(prot.frame) && length(row.names(prot.frame)) == length(unique(row.names(prot.frame))) && all(row.names(prot.frame) == standardize.names(row.names(prot.frame)))
}

read.prot.frame <- function(file)
{
  datf <- read.delim(file = file, row.names = 'filename')
#  datf <- read.delim(file = file, as.is = (names(datf) == 'filename'))
  row.names(datf) <- standardize.names(row.names(datf))
  stopifnot(is.prot.frame(datf))
  datf
}

prot.names <- function(prot.frame, rows)
# rows can be numeric or logical
{
  stopifnot(is.prot.frame(prot.frame))
  stopifnot(is.numeric(rows) || (is.logical(rows) &&length(rows) == nrow(prot.frame)))
  row.names(prot.frame)[rows]
}

is.mass.spectrum <- function(mass.spectrum) {is.data.frame(mass.spectrum) && ncol(mass.spectrum) == 2 && !is.unsorted(mass.spectrum[, 1], na.rm = TRUE)}

as.mass.spectrum <- function(...) {mass.spectrum <- data.frame(...); stopifnot(is.mass.spectrum(mass.spectrum)); mass.spectrum}

save.mass.spectrum <- function(..., relative.path = NULL, file = NULL, row.names = FALSE, path.sep = '')
{ save.spreadsheet(x = as.mass.spectrum(...), sep = ',', col.names = TRUE, row.names = row.names, file = file.or.full.path(relative.path = relative.path, file = file, sep = path.sep))}

read.mass.spectrum <- function(relative.path = NULL, file = NULL, sep = '', ...)
{
  stopifnot(xor(is.null(relative.path), is.null(file)))
  as.mass.spectrum(read.csv(file.or.full.path(relative.path = relative.path, file = file, sep = sep), ...))
}

read.mass.spectra <- function(relative.path = NULL, path = NULL, sep = '', ...)
{
  stopifnot(xor(is.null(relative.path), is.null(path)))
  if(is.null(path)) path <- full.path(relative.path, sep = sep)
  mspec.files <- list.files(path = path, pattern = '*.csv')
  datfs <- lapply(mspec.files, function(smpec.file){read.mass.spectrum(file = if(path == '.') smpec.file else paste(path, smpec.file, sep = sep), sep = sep, ...)})
  lapply(datfs, function(datf){stopifnot(is.mass.spectrum(datf))})
  names(datfs) <- standardize.names(mspec.files)
  mass.spectra <- data.frames.as.array(datfs)
  stopifnot(is.mass.spectra.obj(mass.spectra))
  mass.spectra
}

summary.mass.spectrum <- function(..., FUN = mean)
{
  mass.spectra <- bind.arrays.3d(...)
  if(!all(apply(mass.spectra, 3, function(ms){all.eq(mass.spectra[, 1, 1], ms[, 1])}))) {print('Inconsistent masses.'); browser()}
  mat <- apply(mass.spectra, c(1, 2), FUN)
  stopifnot(length(dim(mat)) == 2)
  stopifnot(dim(mat) == dim(mass.spectra)[1:2])
  as.mass.spectrum(mat)
}

mass.smooth.weight <- 0.99

logmass.spectrum <- function(..., logmass.scale = NULL, smooth.weight = mass.smooth.weight)
{
  mass.spectrum <- as.mass.spectrum(...)
  intensities <- mass.spectrum[, 2]
  logmasses <- log(mass.spectrum[, 1])
#  mass.fun <- approxfun(x = intensities, y = logmasses)
#  datf <- data.frame(sapply(intensities, mass.fun), intensities, row.names = row.names(mass.spectrum))
#  stopifnot(sum(is.na(datf[, 1])) <= sum(is.na(mass.spectrum[, 2]))) # Interpolation should remove any NAs from taking the log of the mass.
  datf <- mass.spectrum
  datf[, 1] <- logmasses
  names(datf) <- c(paste('log', names(mass.spectrum)[1], sep = '.'), names(mass.spectrum)[2])
  if(!is.null(logmass.scale)) datf <- moving.stat(datf, logmass.scale, FUN = median, smooth.weight = smooth.weight)
  if(!is.mass.spectrum(datf)) {print('datf is not a mass spectrum'); browser()}
  datf
}

new.intensity.default <- function(old.intensity, FUN = median)
{
  stopifnot(is.vector(old.intensity))
  loc <- FUN(old.intensity)
  stopifnot(is.finite(loc) && loc > 0)
  old.intensity / loc
}

normalize.mass.spectrum <- function(..., new.intensity = new.intensity.default)
{
  mass.spectrum <- as.mass.spectrum(...)
  ms.names <- names(mass.spectrum)
  mass.spectrum[, 2] <- new.intensity.default(mass.spectrum[, 2])
  names(mass.spectrum) <- ms.names
  stopifnot(is.mass.spectrum(mass.spectrum))
  mass.spectrum
}

normalize.mass.spectra <- function(mass.spectra, new.intensity = new.intensity.default)
{
  stopifnot(is.mass.spectra.obj(mass.spectra))
  ms.list <- lapply(1:(dim(mass.spectra)[3]), function(i) {mass.spectrum.from.spectra(mass.spectra, i)})
  spectra.names <- dimnames(mass.spectra)[[3]]
#  apply(mass.spectra, 3, function(mat){stopifnot(length(dim(mat)) == 2); })
  mass.spectra <- data.frames.as.array(lapply(ms.list, function(ms){normalize.mass.spectrum(ms, new.intensity = new.intensity)}))
  stopifnot(is.null(spectra.names) || dim(mass.spectra)[3] == length(spectra.names))
  dimnames(mass.spectra)[[3]] <- spectra.names
  stopifnot(is.mass.spectra.obj(mass.spectra))
  mass.spectra
}

is.mass.spectra.obj <- function(mass.spectra) {is.array(mass.spectra) && length(dim(mass.spectra)) == 3 && dim(mass.spectra)[2] == 2}

mass.spectrum.from.spectra <- function(mass.spectra, index.or.name)
{
  stopifnot(is.mass.spectra.obj(mass.spectra))
  as.mass.spectrum(mass.spectra[, , index.or.name])
}

extract.mass.spectrum <- function(mass.peaks.obj, indices.col = NULL, with.peaks = TRUE, logmass = FALSE)
{
  datf <- if(with.peaks) mass.peaks.obj$mspec else mass.peaks.obj$mspec.without.peaks
  if(logmass) datf <- logmass.spectrum(datf)
  ms <- if(is.null(indices.col))
    datf
  else
  {
    ind <- mass.peaks.obj$indices
    row.range <- if(is.matrix(ind)) ind[c(1,3), indices.col]
    else
    {
      stopifnot(is.vector(ind) && indices.col == 1)
      ind[c(1, 3)]
    } 
#    ms <- datf
#    ms[, 2] <- NA
    datf[row.range[1]:row.range[2], ]
  }
  as.mass.spectrum(ms)
}

plot.peaks <- function(mass.peaks.obj, logmass = TRUE)
{
  plot(extract.mass.spectrum(mass.peaks.obj, logmass = logmass))
  ind <- mass.peaks.obj$indices
  if(is.vector(ind)) ind <- matrix(ind, ncol = 1)
  lapply(1:ncol(ind), function(i){lines(extract.mass.spectrum(mass.peaks.obj, indices.col = i, logmass = logmass), col = i)})
  NULL
}

new.mass.peaks.obj <- function(intensity, masses, indices, extreme.slope, neglect.abs.slope, slopes, mspec, mspec.without.peaks)
{
  npeaks <- length(intensity)
  if(npeaks > 1)
  {
    if(!all(npeaks == c(ncol(masses), ncol(indices)))) {print('Inconsistent matrix dimensions'); browser()}
    stopifnot(all(3 == c(nrow(masses), nrow(indices))))
    stopifnot(is.null(slopes) || (ncol(slopes) == npeaks && nrow(slopes) + 1 == nrow(mspec)))
  }
  else
  {
    stopifnot(length(masses) == 3 && length(indices) == 3)
    stopifnot(is.null(slopes) || length(slopes) + 1 == nrow(mspec))
  }
  stopifnot(is.null(mspec.without.peaks) || all(dim(mspec) == dim(mspec.without.peaks)))

  list(intensity = intensity, masses = masses, indices = indices, extreme.slope = extreme.slope, neglect.abs.slope = neglect.abs.slope, slopes = slopes, mspec = mspec, mspec.without.peaks = mspec.without.peaks)
}

find.mass.peaks <- function(..., logmass.scale = NULL, max.npeaks = 1, min.abs.slope.p = NULL, neglect.abs.slope.p = min.abs.slope.p, neglect.abs.slope = NULL, smooth.weight = mass.smooth.weight)
{
  stopifnot(is.null(neglect.abs.slope.p) || is.null(neglect.abs.slope))
  stopifnot(!is.null(max.npeaks) || !is.null(min.abs.slope.p))
  if(is.null(max.npeaks) || max.npeaks > 1)
  {
    include.peak <- function(mass.peaks.obj)
    {
      !is.null(mass.peaks.obj) && (is.null(max.npeaks) || peak.i <= max.npeaks) && (is.null(min.abs.slope) || abs(mass.peaks.obj$extreme.slope) >= min.abs.slope)
    }
    peak.i <- 1
    peaks <- list()
    next.peak <- find.mass.peaks(..., logmass.scale = logmass.scale, max.npeaks = 1, neglect.abs.slope.p = neglect.abs.slope.p)
    min.abs.slope <- if(is.null(min.abs.slope.p)) NULL else qdata(abs(next.peak$slopes), min.abs.slope.p)
    while(include.peak(next.peak))
    {
      peaks[[peak.i]] <- next.peak
      peak.i <- peak.i + 1
      next.peak <- find.mass.peaks(next.peak$mspec.without.peaks, logmass.scale = NULL, max.npeaks = 1, neglect.abs.slope = next.peak$neglect.abs.slope)
    }
    stopifnot(length(peaks) > 0)
    peaks.fields <- function(field.name){fields(peaks, field.name)}
    
    return(new.mass.peaks.obj(intensity = peaks.fields('intensity'), masses = peaks.fields('masses'), indices = peaks.fields('indices'), extreme.slope = peaks.fields('extreme.slope'), neglect.abs.slope = peaks.fields('neglect.abs.slope'), slopes = peaks.fields('slopes'), mspec = peaks[[1]]$mspec, mspec.without.peaks = peaks[[length(peaks)]]$mspec.without.peaks))
  }

  stopifnot(max.npeaks == 1) # What follows is used to find a single peak.
  mspec <- as.mass.spectrum(...)
  if(sum(is.finite(mspec[, 2])) <= 3) return(NULL)
  logmspec <- logmass.spectrum(mspec, logmass.scale = logmass.scale, smooth.weight = smooth.weight)

  lms.slopes <- first.differences(logmspec[, 2]) / first.differences(logmspec[, 1])
  if(!is.null(neglect.abs.slope) || !is.null(neglect.abs.slope.p))
  {
    if(is.null(neglect.abs.slope)) neglect.abs.slope <- qdata(abs(lms.slopes), neglect.abs.slope.p)
    make.small.slopes.zero <- function()
    {
      if(is.null(neglect.abs.slope) || !is.finite(neglect.abs.slope) || !is.vector(neglect.abs.slope) || length(neglect.abs.slope) != 1){print('Slope threshold error'); browser()}
      new.slopes <- ifelse(abs(lms.slopes) >= neglect.abs.slope, lms.slopes, 0)
      if(is.null(new.slopes) || is.null(neglect.abs.slope) || !all(abs(new.slopes) >= neglect.abs.slope | new.slopes == 0, na.rm = TRUE)) {print('Slope problem'); browser()}
      new.slopes
    }
    lms.slopes <- make.small.slopes.zero()
  }
  indices.with.peak <- which(abs(lms.slopes) == max(abs(lms.slopes), na.rm = TRUE))
  stopifnot(length(indices.with.peak) > 0)
  index.with.peak <- which(mspec[, 2] == max(mspec[, 2][indices.with.peak]))[1] # element of indices.with.peak with highest intensity
  if(length(indices.with.peak) == 0 || lms.slopes[index.with.peak] == 0) return(NULL) 
  if(!(index.with.peak >= 1 && index.with.peak <= nrow(logmspec))) {print('bad index.with.peak'); browser()}
#  is.neg <- function(x){!is.na(x) && x < 0}
#  is.pos <- function(x){!is.na(x) && x > 0}
  is.nonneg <- function(x){!is.na(x) && x >= 0}
  is.nonpos <- function(x){!is.na(x) && x <= 0}
  left.and.top.indices <- function(intensities, slopes, pos.slope.index)
  {
    stopifnot(length(intensities) == length(slopes) + 1)
    if(!is.nonneg(slopes[pos.slope.index])) {print('Slope is not positive'); browser()}
    left.index <- function(current.index)
    {
      if(!(current.index >= 1 && current.index <= nrow(logmspec))) {print('decreasing current.index out of range'); browser()}
      while(current.index != 1 && is.nonneg(slopes[current.index - 1]))
        current.index <- current.index - 1
      current.index
    }
    top.index <- function(current.index)
    {
      if(!(current.index >= 1 && current.index <= nrow(logmspec))) {print('increasing current.index out of range'); browser()}
      if(current.index == length(slopes) + 1)
        current.index
      else
      {
        while(current.index != length(slopes) && is.nonneg(slopes[current.index + 1]))
          current.index <- current.index + 1
        current.index + 1
      }
    }
    left.i.0 <- left.index(pos.slope.index)
    top.i.0 <- top.index(pos.slope.index)
    if(top.i.0 < left.i.0) {print('bad left.i.0 or top.i.0'); browser()}
    if(top.i.0 == left.i.0) warning('Peak is on the edge of the spectrum')
    c(left.i.0, top.i.0)
  }
  right.and.top.indices <- function(intensities, slopes, neg.slope.index)
  {
    if(!is.nonpos(slopes[neg.slope.index])) {print('Slope is not negative'); browser()}
    rev(left.and.top.indices(-intensities, -slopes, neg.slope.index))
  }
  if(lms.slopes[index.with.peak] >= 0)
  {
    lati <- left.and.top.indices(logmspec[, 2], lms.slopes, index.with.peak)
    slope.index <- lati[2]
#if(lati[2] == 100) {print('temp1'); browser()}
    rati <- if(slope.index <= nrow(logmspec) && is.finite(lms.slopes[slope.index]))
        right.and.top.indices(logmspec[, 2], lms.slopes, slope.index)
      else
        rep(lati[2], 2)
  }
  else
  {
    rati <- right.and.top.indices(logmspec[, 2], lms.slopes, index.with.peak)
    slope.index <- rati[2] - 1
#if(rati[2] == 100) {print('temp2'); browser()}
    lati <- if(slope.index >= 1 && is.finite(lms.slopes[slope.index]))
        left.and.top.indices(logmspec[, 2], lms.slopes, slope.index)
      else
        rep(rati[2], 2)
  }
  left.i <- lati[1]
#  top.i <- lati[2]
#  if(rati[2] != top.i) {print('Inconsistent maxima'); browser()}
  right.i <- rati[1]
  if(right.i <= left.i) {print('Bad edges of peak'); browser()}
  get.top.i <- function(index1, index2)
  {
    small.i <- min(index1, index2)
    big.i <- max(index1, index2)
    intensities <- logmspec[, 2]
    is.in.range <- sapply(1:length(intensities), function(i){i >= small.i & i <= big.i})
    top.i <- which(intensities == max(intensities[is.in.range]) & is.in.range)[1]
    stopifnot(is.finite(top.i) && length(top.i) == 1)
    if(top.i < small.i || top.i > big.i) {print('bad top.i'); browser()}
#if(top.i == 100) {print('temp3'); browser()}
    top.i
  }
  top.i <- get.top.i(lati[1], rati[1])
  if(!is.finite(top.i) || length(top.i) != 1) {print('top.i error'); browser()}
  if(top.i < left.i || top.i > right.i) {print('top.i out of range'); browser()}
  new.mspec.nopeak <- new.mspec <- data.frame(exp(logmspec[, 1]), logmspec[, 2])
  new.mspec.nopeak[left.i:right.i, 2] <- NA
  names(new.mspec.nopeak) <- names(new.mspec) <- names(mspec)
  new.mass.peaks.obj(intensity = mspec[top.i, 2], masses = c(mspec[left.i, 1], mspec[top.i, 1], mspec[right.i, 1]), indices = c(left.i, top.i, right.i), extreme.slope = lms.slopes[index.with.peak], neglect.abs.slope = neglect.abs.slope, slopes = lms.slopes, mspec = as.mass.spectrum(new.mspec), mspec.without.peaks = as.mass.spectrum(new.mspec.nopeak))
}

get.intensity <- function(mass.spectrum, mass.index){stopifnot(is.mass.spectrum(mass.spectrum)); mass.spectrum[mass.index, 2]}

mass.peaks.from.indices <- function(mass.spectrum, index.mat, extreme.slope, neglect.abs.slope, mspec.without.peaks)
{
  stopifnot(is.mass.spectrum(mass.spectrum))
  stopifnot(is.matrix(index.mat))
  stopifnot(nrow(index.mat) == 3)
  masses2 <- apply(index.mat, 2, function(column){mspec[column, 1]})
  if(nrow(masses2) != 3) {print('bad masses2'); browser()}
  
  new.mass.peaks.obj(intensity = apply(index.mat, 2, function(column){get.intensity(mass.spectrum, column[2])}), masses = masses2, indices = index.mat, extreme.slope = extreme.slope, neglect.abs.slope = neglect.abs.slope, slopes = NULL, mspec = mass.spectrum, mspec.without.peaks = mspec.without.peaks)
}


combine.peaks <- function(mass.peaks.obj, min.mass.ratio = 1.001, is.same.peak = function(mass1, mass2){is.finite(mass1) && is.finite(mass2) && max(mass1, mass2) / min(mass1, mass2) <= min.mass.ratio})
{
# print(is.same.peak(mass.peaks.obj[1], mass.peaks.obj[2]))
  attach(mass.peaks.obj)
  stopifnot(length(intensity) > 1)
  combined.indices <- function(indices0)
  {
    stopifnot(is.matrix(indices0))
    get.mass <- function(peak.index) {mass.peaks.obj$mspec[indices0[2, peak.index], 1]}
    combine.pair <- function(ind.mat, peak.index1, peak.index2)
    {
      dim0 <- dim(ind.mat)
      combined.column <- function(vec1, vec2)
      {
        stopifnot(length(vec1) == 3 && length(vec2) == 3)
        vec <- c(min(vec1[1], vec2[1]), if(get.intensity(mass.peaks.obj$mspec, vec1[2]) > get.intensity(mass.peaks.obj$mspec, vec2[2])) vec1[2] else vec2[2], max(vec1[3], vec2[3]))
        stopifnot(all(is.finite(vec)) && vec[1] <= vec[2] && vec[2] <= vec[3])
        vec
      }
      ind.mat[, c(peak.index1, peak.index2)] <- cbind(combined.column(ind.mat[, peak.index1], ind.mat[, peak.index2]), c(NA, NA, NA))
      stopifnot(dim(ind.mat) == dim0)
      ind.mat
    }
    for(peak.i in 1:(ncol(indices0) - 1))
    {
      i1 <- peak.i; i2 <- peak.i + 1
      mass1 <- get.mass(i1)
      mass2 <- get.mass(i2)
      if(is.same.peak(mass1, mass2)) indices0 <- combine.pair(indices0, i1, i2)
    }
    peak.is.present <- is.finite(indices0[2, ])
    if(all(peak.is.present))
      indices0
    else
      combined.indices(indices0[, peak.is.present])
  }

  mass.peaks.from.indices(mass.spectrum = mspec, index.mat = combined.indices(indices), extreme.slope = extreme.slope, neglect.abs.slope = neglect.abs.slope, mspec.without.peaks = mspec.without.peaks)
}

derivative.mass.peaks <- function(mass.spectrum, mass.peaks.obj)
{
#  index.ranges <- apply(mass.peaks.obj$indices, 2, function(column){column[c(1, 3)]})
  stopifnot(is.mass.spectrum(mass.spectrum))
  attach(mass.peaks.obj)
  inds <- indices
  top.ind <- function(ind.column)
  {
    intensity.vec <- mass.spectrum[, 2]
    is.in.range <- sapply(1:length(intensity.vec), function(i){i >= ind.column[1] & i <= ind.column[3]})
    t.i <- which(is.in.range & max(intensity.vec[is.in.range]) == intensity.vec)
    stopifnot(is.finite(t.i) && length(t.i) >= 0)
    if(length(t.i) > 1) warning('Intensity tie broken arbitrarily')
    t.i[1]
  }
  top.inds <- as.vector(apply(inds, 2, top.ind))
  stopifnot(length(top.inds) == ncol(inds))
  inds[2, ] <- top.inds
  apply(inds, 2, function(column){stopifnot(column[1] <= column[2] && column[2] <= column[3])})
  mass.peaks.from.indices(mass.spectrum = mass.spectrum, index.mat = inds, extreme.slope = extreme.slope, neglect.abs.slope = neglect.abs.slope, mspec.without.peaks = NULL)
}

num.mass.peaks <- function(mass.peaks.obj)
{
  ind <- mass.peaks.obj$indices
  if(is.matrix(ind)) ncol(ind) else 1
}

is.intensities.and.masses.obj <- function(intensities.and.masses.obj)
{
  attach(intensities.and.masses.obj)
  all(dim(intensity.max) == dim(mass)) && !is.null(dimnames(mass)[[2]]) && all(dimnames(intensity.max)[[2]] == dimnames(mass)[[2]])
}

new.intensities.and.masses.obj <- function(intensity.max, mass)
{
  iam <- list(intensity.max = intensity.max, mass = mass)
  stopifnot(is.intensities.and.masses.obj(iam))
  iam
}

intensities.and.masses <- function(mass.spectra, mass.peaks.obj)
{
  stopifnot(is.mass.spectra.obj(mass.spectra))
  mass.peaks.objects <- lapply(1:(dim(mass.spectra)[3]), function(i){derivative.mass.peaks(mass.spectrum = mass.spectrum.from.spectra(mass.spectra = mass.spectra, index.or.name = i), mass.peaks.obj = mass.peaks.obj)})
  intensity.max = sapply(mass.peaks.objects, function(mpo){mpo$intensity})
  mass = sapply(mass.peaks.objects, function(mpo){mpo$masses[2, ]})
  stopifnot(all(dim(intensity.max) == dim(mass)))
  stopifnot(length(mass.peaks.objects) == ncol(mass))
  spectra.names <- standardize.names(dimnames(mass.spectra)[[3]])
  stopifnot(is.null(spectra.names) || length(spectra.names) == ncol(intensity.max))
  colnames(intensity.max) <- colnames(mass) <- spectra.names
  iam <- new.intensities.and.masses.obj(intensity.max = intensity.max, mass = mass)
  stopifnot(is.intensities.and.masses.obj(iam))
  iam
}

lm.prot.frame <- function(indep.names, intensities.and.masses.obj, prot.frame, fun.name = 'lm', transform.fun = NULL, univariate = FALSE)
{
  intensity <- t(intensities.and.masses.obj$intensity.max)
  stopifnot(nrow(intensity) <= nrow(prot.frame))
  if(is.function(transform.fun)) intensity <- transform.fun(intensity)
  selected.names <- dimnames(intensity)[[1]]
  if(is.null(selected.names)) stop('intensities.and.masses.obj lacks names')
  prot.frame <- prot.frame[selected.names, ]
  stopifnot(nrow(intensity) == nrow(prot.frame))
  stopifnot(all(selected.names == row.names(prot.frame)))
  right.hand.side <- function(...){paste(..., sep = '+')}
  lm.eval.arg <- function(dep.variable.name, print.cmd)
  {
    eval.arg(fun.name, '(formula = ', dep.variable.name, ' ~ ', do.call('right.hand.side', as.list(indep.names)), ', prot.frame)', print.cmd = print.cmd)
  }
  if(univariate)
  {
    lm.univariate <- function(peak.i)
    {
      intensity.vector <- intensity[, peak.i]
      eval(lm.eval.arg(dep.variable.name = 'intensity.vector', print.cmd = (peak.i == 1)))
    }
    lapply(1:ncol(intensity), lm.univariate)
  }
  else
     eval(lm.eval.arg(dep.variable.name = 'intensity', print.cmd = TRUE))
}

aov.prot.frame <- function(indep.names, intensities.and.masses.obj, prot.frame, ...)
{
  lm.prot.frame(indep.names = indep.names, intensities.and.masses.obj = intensities.and.masses.obj, prot.frame = prot.frame, fun.name = 'aov', ...)
}
  

select.intensities.and.masses <- function(intensities.and.masses.obj, prot.frame, prot.rows)
# prot.rows can be numeric or logical
{
  if(!is.intensities.and.masses.obj(intensities.and.masses.obj)) {print('intensities.and.masses.obj has problems'); browser()}
  attach(intensities.and.masses.obj)
  stopifnot(is.prot.frame(prot.frame))
  stopifnot(nrow(prot.frame) >= ncol(mass))
  pnames <- prot.names(prot.frame = prot.frame, rows = prot.rows)
  if(!all(pnames %in% dimnames(mass)[[2]])) {print('Mismatch betweeen names of intensities.and.masses.obj and prot.frame'); browser()}
  new.intensities.and.masses.obj(intensity.max = intensity.max[, pnames], mass = mass[, pnames])
}

tapply.intensities.and.masses <- function(intensities.and.masses.obj, prot.frame, variable.names, prot.rows = NULL, FUN = mean)
{
  if(!is.null(prot.rows)) return(tapply.intensities.and.masses(select.intensities.and.masses(intensities.and.masses.obj, prot.frame = prot.frame, prot.rows = prot.rows), prot.frame = prot.frame, prot.rows = NULL, variable.names = variable.names, FUN = FUN))

  stopifnot(is.intensities.and.masses.obj(intensities.and.masses.obj))
  attach(intensities.and.masses.obj)
  stopifnot(is.prot.frame(prot.frame))
  stopifnot(nrow(prot.frame) >= ncol(mass))
  prot.frame <- prot.frame[dimnames(mass)[[2]], ]
  stopifnot(nrow(prot.frame) == ncol(mass))
  facs <- lapply(variable.names, function(var.name) {prot.frame[, var.name]})
  facs <- list(factor(as.vector(facs[[1]]))) # level names are lost
  stopifnot(all(sapply(facs, length) == ncol(mass)))
  new.mat <- function(mat) {t(apply(mat, 1, function(mat.row){tapply(mat.row, INDEX = facs, FUN = FUN)}))}
  int2 <- new.mat(intensity.max)
  mass2 <- new.mat(mass)
  if(length(dim(int2)) != length(dim(mass)) || !all(dim(int2) <= dim(mass))) {print('matrix dimension error'); browser()}
  new.intensities.and.masses.obj(intensity.max = int2, mass = mass2)
}

wilcox.cdf <- function(w, intensities.and.masses.1, intensities.and.masses.2)
{
  pwilcox(w, n = ncol(intensities.and.masses.1$mass), m = ncol(intensities.and.masses.2$mass))
}

wilcox.quantile <- function(p, intensities.and.masses.1, intensities.and.masses.2)
{
  qwilcox(p, n = ncol(intensities.and.masses.1$mass), m = ncol(intensities.and.masses.2$mass))
}

wilcox.median <- function(intensities.and.masses.1, intensities.and.masses.2)
{ wilcox.quantile(p = .5, intensities.and.masses.1, intensities.and.masses.2)}

prot.h0dist.diff.tests <- function(intensities.and.masses.1, intensities.and.masses.2, null.cdf = function(stat){wilcox.cdf(w = stat + wilcox.median(intensities.and.masses.1, intensities.and.masses.2), intensities.and.masses.1, intensities.and.masses.2)}, alternative = c('two.sided', 'less', 'greater'), test.stat = function(x, y){wilcox.test(x, y)$statistic - wilcox.median(intensities.and.masses.1, intensities.and.masses.2)}, replace = FALSE, balance = FALSE, desum = FALSE, regress.file = '', nboot.rescale = 0, diff.scale.estimator = NULL, scale.file = '', test.name = 'prot.h0dist.diff.tests', file.base = test.name, ...)
{
  stopifnot(is.intensities.and.masses.obj(intensities.and.masses.1) && is.intensities.and.masses.obj(intensities.and.masses.2))
  mass1 <- intensities.and.masses.1$mass; mass2 <- intensities.and.masses.2$mass
  intensity1 <- intensities.and.masses.1$intensity; intensity2 <- intensities.and.masses.2$intensity
  stopifnot(nrow(mass1) == nrow(mass2))
  mean.masses <- function(mass.mat) {apply(mass.mat, 1, mean)}
  are.masses.compatible <- function() {all(order(mean.masses(mass1)) == order(mean.masses(mass2)))}
  stopifnot(are.masses.compatible())
  
  datf <- h0dist.diff.tests(two.samples = list(intensity1, intensity2), null.cdf = null.cdf, alternative = alternative, test.stat = test.stat, replace = replace, balance = balance, desum = desum, regress.file = regress.file, nboot.rescale = nboot.rescale, diff.scale.estimator = diff.scale.estimator, scale.file = scale.file, test.name = test.name, file.base = file.base, ...)
  mass.mean <- mean.masses(cbind(mass1, mass2))
  stopifnot(nrow(datf) == length(mass.mean))
  datf <- data.frame(mass.mean = mass.mean, datf)
  if(saving.results(file.base)) save.err.rates.table(datf, file.base = file.base)
  datf
}


