# FDRCI, Copyright (c) 2004 by David R. Bickel.
# Last modified by David R. Bickel on 3/27/04. 
# Created by David R. Bickel on 3/19/04. 

source('multtests.r')

wilcox.p.values <- function(mat1, mat2, b = 0, size = NULL, replace = TRUE, ...)
{
  if(b > 1)
    return(sapply(1:b, function(i){wilcox.p.values(mat1, mat2, b = 1, size = size, replace = replace, ...)}))
  if(b > 0)
  {
    sc <- function(mat){sample.columns(mat, replace = replace, size = if(is.null(size)) ncol(mat) else size)}
    mat1 <- sc(mat1)
    mat2 <- sc(mat2)
  }
  else
    print('not resampled')
  stopifnot(nrow(mat1)==nrow(mat2))
  pv <- numeric(nrow(mat1))
  for(i in 1:nrow(mat1)) pv[i] <- wilcox.test(mat1[i,], mat2[i,], ...)$p.value
  pv
}

rej <- function(p.values, alpha = .01, med = NULL)
{
  if(!is.matrix(p.values))
    p.values <- matrix(p.values, ncol = 1)
  if(is.null(med))
    apply(p.values, 2, function(co){sum(co <= alpha)})
  else
  {
    alphas <- apply(p.values, 2, function(co){sort(co)[med]})
    alpha <- median(alphas)
    cool.cat('alpha == ', alpha)
    rej(p.values, alpha = alpha, med = NULL)
  }
}

FDR <- function(portion.h0.true = NULL, alpha, portion.reject)
{
  if(portion.reject == 0)
    NA
  else
    min(1, (if(is.null(portion.h0.true)) 1 else portion.h0.true) * alpha / portion.reject)
}

is.missing <- function(x) {is.null(x) || any(is.na(x))}

plot.FDR.CI <- function(p.values, hurst = seq(0.5, 1, .01), ylim = c(0, 1), ...)
{
  if(is.matrix(p.values)) stop('p.values cannot be a matrix')
  ci <- FDR.CI(p.values = p.values, hurst = hurst, mult.CIs = TRUE, ...)
  plot(hurst, ci[2, ], xlab = 'H', ylab = 'FDR', ylim = if(is.null(ylim)) c(min(ci), max(ci, na.rm = TRUE)) else ylim)
  lines(hurst, ci[1, ])
  lines(hurst, ci[3, ])
  ci
}

FDR.CI <- function(p.values, sampled.p.values = NULL, portion.h0.true = NULL, alpha = .01, med = NULL, hurst = 0.7, mult.CIs = length(hurst) > 1, prob.reject = NULL, confidence = .95, browse.on.na = FALSE, print.info = TRUE) # originally, hurst = seq(0.5, 1, .01), mult.CIs = FALSE
{
  if(!is.null(med)) stop('calls to FDR are not yet based on alpha from med')
  stopifnot(!is.null(p.values))
  if(mult.CIs && length(hurst) > 1)
  {
    if(print.info) 
    {
      cool.cat('matrix return value')
      FDR.CI(p.values = p.values, sampled.p.values = sampled.p.values, portion.h0.true = portion.h0.true, alpha = alpha, med = med, hurst = hurst[1], mult.CIs = FALSE, prob.reject = prob.reject, confidence = confidence, browse.on.na = browse.on.na, print.info = TRUE)
    }
    return(sapply(hurst, function(h){FDR.CI(p.values = p.values, sampled.p.values = sampled.p.values, portion.h0.true = portion.h0.true, alpha = alpha, med = med, hurst = h, mult.CIs = FALSE, prob.reject = prob.reject, confidence = confidence, browse.on.na = browse.on.na, print.info = FALSE)}))
  }
  if(is.matrix(p.values))
  {
    if(print.info) print('using resample p-values to estimate sd')
    r <- rej(p.values, alpha = alpha, med = med)
    stopifnot(length(r) == ncol(p.values))
    print(cv(r))
    portion.reject <- if(is.null(sampled.p.values))
      mean(r) / nrow(p.values)
    else
    {
      stopifnot(length(sampled.p.values) == nrow(p.values))
      rej(sampled.p.values, alpha = alpha, med = med) / length(sampled.p.values) # sum(sampled.p.values <= alpha) / length(sampled.p.values)
    }
    prob.success <- if(is.null(prob.reject)) portion.reject else prob.reject
    pr.sds <- sd(r) / nrow(p.values)
    cool.cat('Hurst exponent estimate: ', 1 - log(sqrt(prob.success * (1 - prob.success)) / pr.sds) / log(nrow(p.values)))
#    cool.cat('H estimate: ', 1 - log(sqrt(prob.success * (nrow(p.values) - prob.success) / nrow(p.values)) / pr.sds) / log(nrow(p.values)))
  }
  else
  {
    stopifnot(is.null(sampled.p.values))
    if(print.info) print('using H to get sd')
    if(any(hurst < 0 | hurst > 1)) stop("bad hurst")
    portion.reject <- if(length(p.values) == 1)
    {
      m <- p.values
      prob.reject
    }
    else
    {
      m <- length(p.values)
      rej(p.values, alpha = alpha, med = med) / length(p.values)
    }
    stopifnot(m > 0)
    stopifnot(!is.null(portion.reject))
    prob.success <- if(is.null(prob.reject)) portion.reject else prob.reject
    pr.sds <- sqrt((1 - prob.success) * prob.success) / m ^ (1 - hurst)    
  }
  if(print.info) cool.cat('portion.reject == ', portion.reject)
  p <- (1 - confidence) / 2
  pr.lo <- mean(pmax(0, qnorm(p = p, mean = portion.reject, sd = pr.sds)))
  pr.hi <- mean(pmin(1, qnorm(p = p, mean = portion.reject, sd = pr.sds, lower.tail = FALSE)))
  dfdr <- FDR(portion.h0.true = portion.h0.true, alpha = alpha, portion.reject = portion.reject)
  if(is.missing(pr.lo)) {print("pr.lo problem"); browser()}
  dfdr.hi <- FDR(portion.h0.true = portion.h0.true, alpha = alpha, portion.reject = pr.lo)
  if(is.missing(pr.hi)) {print("pr.hi problem"); browser()}
  dfdr.lo <- FDR(portion.h0.true = portion.h0.true, alpha = alpha, portion.reject = pr.hi)
  ci <- c(dfdr.lo, dfdr, dfdr.hi)
  #if(ci[1]==ci[3]) {print("bad ci"); browser()}
  if(browse.on.na && any(is.na(ci))) browser()
  ci
}